// Daniel Shiffman
// Kinect Point Cloud example
// http://www.shiffman.net
// https://github.com/shiffman/libfreenect/tree/master/wrappers/java/processing

import org.openkinect.*;
import org.openkinect.processing.*;
  
// Kinect Library object
Kinect kinect;

float a = 0; // rotateY()
float b = 0; // rotateX()
float c = 0; // rotateZ()
  
float deg = 45; // Start at 45 degrees

// Size of kinect image
int w = 640;
int h = 480;
// int w = 1024;
// int h = 768;

// We'll use a lookup table so that we don't have to repeat the math over and over
float[] depthLookUp = new float[2048];

void setup() {
  size(1024,768,P3D);
  kinect = new Kinect(this);
  kinect.start();
  kinect.enableDepth(true);
  // We don't need the grayscale image in this example
  // so this makes it more efficient
  kinect.processDepthImage(false);
  // kinect.tilt(deg);

  // Lookup table for all possible depth values (0 - 2047)
  for (int i = 0; i < depthLookUp.length; i++) {
    depthLookUp[i] = rawDepthToMeters(i);
  }
  initOSC();
}

void draw() {

  background(0);
  fill(255);
  textMode(SCREEN);
  text("Kinect FR: " + (int)kinect.getDepthFPS() + "\nProcessing FR: " + (int)frameRate,10,16);
  // System.out.println("hi RWH in depthimage() "+fader[1] );

  // Get the raw depth as array of integers
  int[] depth = kinect.getRawDepth();

  // kinect.tilt(rotary[1]*180-90); 
  kinect.tilt((1-xyPad[2].x)*180-90);
  
  // We're just going to calculate and draw every 4th pixel (equivalent of 160x120)
  int skip = 2;

  // Translate and rotate
  // translate(width/2,height/2,-50);
  translate(width/2,height/2,-500*fader[6]-200);
  rotateY(a);
  rotateX(b);
  // rotateZ(c);
  // rotateX(xyPad[1].x);
  // rotateY(xyPad[1].y);
  
  for(int x=0; x<w; x+=skip) {
    for(int y=0; y<h; y+=skip) {
      int offset = x+y*w;
  
      // Convert kinect data to world xyz coordinate
      int rawDepth = depth[offset];
      PVector v = depthToWorld(x,y,rawDepth);

      stroke(255);  // white 
      // stroke("blue"); 
      // stroke(235, 153, 77); // orange 
      // stroke(28, 153, 77); // green 
      // stroke(value1, value2, value3, alpha); 
      // RWH: ok, the Fader1 is value1, Fader2 is value2, Fader3 is value3, Fader4 is alpha
      stroke(fader[1]*255, fader[2]*255, fader[3]*255, fader[4]*255); 
      
      pushMatrix();
      // Scale up by 200
      // float factor = 300;
      float factor = fader[5]*600;
      translate(v.x*factor,v.y*factor,factor-v.z*factor);
      // Draw a point
      // point(0,0);
      
      // replace i -> x; j -> y 
      float x1 = (x+0) * v.x*factor + (y+0);
      float y1 = (x+0) * v.y*factor + (y+0);
      float z1 = (x+0) * factor-v.z*factor + (y+0);

      float x2 = (x+1) * v.x*factor + (y+0);
      float y2 = (x+1) * v.y*factor + (y+0);
      float z2 = (x+1) * factor-v.z*factor + (y+0);
      
      float x4 = (x+0) * v.x*factor + (y+1);
      float y4 = (x+0) * v.y*factor + (y+1);
      float z4 = (x+0) * factor-v.z*factor + (y+1);
      
      // stroke( 40, 100, 134, 1);  
      noFill();
      beginShape(); vertex(x4, y4, z4); vertex(x1, y1, z1); vertex(x2, y2, z2); endShape();
      
      popMatrix();
    }
  }

  // Rotate
  //  a += 0.020f;
  //  b += 0.020f;
  c += 0.020f;
  
  a += xyPad[1].y/30.;
  b += xyPad[1].x/30.;
}

// These functions come from: http://graphics.stanford.edu/~mdfisher/Kinect.html
float rawDepthToMeters(int depthValue) {
  if (depthValue < 2047) {
    return (float)(1.0 / ((double)(depthValue) * -0.0030711016 + 3.3309495161));
  }
  return 0.0f;
}

PVector depthToWorld(int x, int y, int depthValue) {

  final double fx_d = 1.0 / 5.9421434211923247e+02;
  final double fy_d = 1.0 / 5.9104053696870778e+02;
  final double cx_d = 3.3930780975300314e+02;
  final double cy_d = 2.4273913761751615e+02;

  PVector result = new PVector();
  double depth =  depthLookUp[depthValue];//rawDepthToMeters(depthValue);
  result.x = (float)((x - cx_d) * depth * fx_d);
  result.y = (float)((y - cy_d) * depth * fy_d);
  result.z = (float)(depth);
  return result;
}

void stop() {
  kinect.quit();
  super.stop();
}

